/*---------------------------------------------------------------------------+
| PMSPYDLL.C                                                                 |
+----------------------------------------------------------------------------+
|                                                                            |
| Program to spy on other windows message queues                             |
|                                                                            |
| Notes:                                                                     |
|                                                                            |
| 1. This code MUST be in a DLL since we "hook" into the system queue!       |
|                                                                            |
| 2. There are TWO types of routines in this module:                         |
|                                                                            |
|    a) Routines called directly by PM...these are the "hook" routines       |
|                                                                            |
|       SpyInputHookProc                                                     |
|       SpySendMsgHookProc                                                   |
|                                                                            |
|    b) Routines called only by PMSPY instances                              |
|                                                                            |
|       SpyRegister                                                          |
|       SpyDeRegister                                                        |
|                                                                            |
|       SpySetTarget                                                         |
|       SpyUnSetTarget                                                       |
|       SpyQueryTargetWindow                                                 |
|       SpyQueryTargetQueue                                                  |
|                                                                            |                           |
|       SpySetTargetIsWindow                                                 |
|       SpyQueryTargetIsWindow                                               |
|                                                                            |                           |
|       SpySetTargetMsgRange                                                 |
|                                                                            |
|       SpyDllVersion                                                        |
|       SpyQueryDllRegisterMax                                               |
|                                                                            |
+-------------------------------------+--------------------------------------+
|                                     |   Juerg von Kaenel (JVK at ZURLVM1)  |
| Version: 0.02                       |   IBM Research Laboratory            |
|                                     |   Saeumerstrasse 4                   |
|                                     |   CH - 8803 Rueschlikon              |
|                                     |   Switzerland                        |
+-------------------------------------+--------------------------------------+
| History:                                                                   |
| --------                                                                   |
|                                                                            |
| created: jan  6 1989 by J. von Kaenel                                      |
| updated: jan 21 1989 by J. von Kaenel - added support for 2 simultaneous   |
|                                         instances of PMSPY                 |
|          dec 20 1989 smd              - added support for N instances      |
+---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------+
| Includes                                                                   |
+---------------------------------------------------------------------------*/

#include "pmspy.h"

/**************************************************************************
* Local data structures, macros, etc
**************************************************************************/

typedef struct
{
  /**************************************************************************
  * Control Flag: TRUE --> this item is actively SPYing
  *               FALSE--> this item marks end of the variable length list
  **************************************************************************/

  BOOL      fSpyingActive;

  /**************************************************************************
  * Define who we're SPYing on
  **************************************************************************/

  HWND      hwndTarget;                 // which Window
  HMQ        hmqTarget;                 // which Queue
  BOOL      fTargetIsWindow;            // SPYing on Window (vs. Queue)

  /**************************************************************************
  * Additional control data
  **************************************************************************/

  HAB        habSpy;                    // HAB of destination SPY window
  HWND      hwndSpy;                    // destination SPY window

  HMODULE   modPFN;                     // DLL handle

  MSG       msgLow,                     // MSG range to pass on to SPYee
            msgHigh;

  /**************************************************************************
  * QMSG data for passing onto PMSPY instances
  *
  * - we pass 'normalized' data to each instance regardless of "hook" source
  * - NOTE: we can do this because each message is really 'single-threaded'
  *         through this DLL because WinSendMsg is used to: (a) invoke
  *         the "hook" routiens and (b) invoke PMSPY to 'spy' the MSG
  **************************************************************************/

  QMSG      qMsg;

} dllSPY, near * PdllSPY;

/**************************************************************************
* Macros to generate initialized Spys[] data elements
**************************************************************************/

#define spyEOT {FALSE,                          \
                SPY_BAD_HWND,SPY_BAD_HMQ,FALSE, \
                SPY_BAD_HAB,SPY_BAD_HWND,NULL,  \
                0,0xFFFF}

#define spyDEF {TRUE,                           \
                SPY_BAD_HWND,SPY_BAD_HMQ,FALSE, \
                SPY_BAD_HAB,SPY_BAD_HWND,NULL,  \
                0,0xFFFF}

#define ID_NOT_DEFINED   0xFFFF

/**************************************************************************
* Macro to validate a specified SPY instance ID
*
* - if the ID is not valid, a RETURN is generated using the specified value
* - if the ID is     valid, set pointer to ID's data
*
* - Tests for Validity:
*   a) ID is less than current 1-origin count of PMPSY instances
*   b) ID is defined in Id2Spys[]

**************************************************************************/

#define IF_BAD_ID_RETURN_ELSE_SET(idSpy,retValue,pSpy)               \
if ( (idSpy < uSpyInstances) && (Id2Spys[idSpy] != ID_NOT_DEFINED) ) \
  pSpy = &Spys[ Id2Spys[idSpy] ];                                    \
else                                                                 \
  return(retValue);

/**************************************************************************
* This prevents "C" from adding it's own initialization code!!
*
* (and hauling in lots of runtime support code)
*
**************************************************************************/

int _acrtused = 0;

/**************************************************************************
* Localized DLL data for controlling using PMSPY instances
**************************************************************************/

static USHORT uSpyInstances = 0;             // # active SPYee(s)

/**************************************************************************
* Translate a "PMSPY instance ID" (0..n) to it's supporting data
**************************************************************************/

static USHORT near Id2Spys[] =
{
  ID_NOT_DEFINED,       // Instance #1 data
  ID_NOT_DEFINED,       // Instance #2 data
  ID_NOT_DEFINED,       // Instance #3 data
  ID_NOT_DEFINED,       // Instance #4 data
  ID_NOT_DEFINED,       // Instance #5 data
  ID_NOT_DEFINED,       // Instance #6 data
  ID_NOT_DEFINED,       // Instance #7 data
  ID_NOT_DEFINED        // Instance #8 data
};

#define MAX_SPYEES    ( sizeof(Id2Spys) / sizeof(Id2Spys[0]) )

/**************************************************************************
* Supporting SPY data
*
* Implementation Notes:
* ---------------------
* 1) because this table is searched sequentially each time one of our
*    "hook" procedures is called to handle a MSG, we want to only search
*    as many instances as we have currently active SPYees.  So, we keep
*    table 'packed' as tight as possible.  By that, i mean that any time
*    a PMSPY instance DeRegisters, we move all data elements below it
*    up one slot.  Within a "hook" procedure we only 'look' at the first
*    N items (where N is the number of active SPYees)
*
* 2) to maintain the SPYEE value returned when a PMPSY instance Registers,
*    we also maintain Id2Spys[].
*
**************************************************************************/

static dllSPY near Spys[] =
{
  spyEOT,               // room for Instance #1
  spyEOT,               // room for Instance #2
  spyEOT,               // room for Instance #3
  spyEOT,               // room for Instance #4
  spyEOT,               // room for Instance #5
  spyEOT,               // room for Instance #6
  spyEOT,               // room for Instance #7
  spyEOT,               // room for Instance #8

  spyEOT                // permanent End-Of-Table marker
};

/**************************************************************************
* Initialized dllSPY items for adding/deleting Spys[] items
**************************************************************************/

static const dllSPY defSPY = spyDEF;
static const dllSPY defEOT = spyEOT;

/*---------------------------------------------------------------------------+
| Input Hook procedure
|
| This procedure gets calls after a MSG is removed from an application's
| queue before being returned by a WinGetMsg or WinPeekMsg
|
| Notes:
| ------
| 1) we are called by PM, not a PMSPY instance.  This implies that we
|    don't know which PMSPY instance, if any, should SPY this message.
|    That's why we have to scan the complete Spys[] table and route it
|    to all interested SPYees...
| 2) we always return FALSE so that any other hooks "downstream"
|    can also have a chance at this message
|
| Reference: PM Programming Reference: Volume Two,
|            Chapter 10 - "Functions Supplied by Application"
|
+---------------------------------------------------------------------------*/

BOOL EXPENTRY SpyInputHookProc(HAB   habSpy_NotUsed,
                               PQMSG pQmsg,
                               BOOL  bRemove_NotUsed)
{
  register PdllSPY pSpy;

  /**************************************************************************
  * Check each of the 'active' SPYees to see if interested in this message
  **************************************************************************/

  for(/* Initialize */ pSpy = &Spys[0];      // Start @ first Spyee
      /* While      */ pSpy->fSpyingActive;  // Not at End-Of-List
      /* Iterate    */ pSpy++)               // Move to next Spyee
  {
    /**************************************************************************
    * Determine if this MSG should go to this "active" SPYee:
    *
    * 1) destined for the correct Window or Queue
    * 2) its a MSG within the desired range
    **************************************************************************/

    if ( (pSpy->fTargetIsWindow
          ? pSpy->hwndTarget == pQmsg->hwnd
          : pSpy-> hmqTarget == (HMQ)WinQueryWindowULong(pQmsg->hwnd,QWL_HMQ)
         ) &&
         ((pQmsg->msg >= pSpy->msgLow) && (pQmsg->msg <= pSpy->msgHigh)) )
    {
      pSpy->qMsg = *pQmsg;                    // copy QMSG data

      WinSendMsg(pSpy->hwndSpy,
                 MSG_FROM_SPY_HOOK,
                 MPFROMP(&pSpy->qMsg),
                 NULL);
    }
  }

  /**************************************************************************
  * Indicate "pass the message to the next hook"
  **************************************************************************/

  return(FALSE);
}

/*---------------------------------------------------------------------------+
| SendMsg Hook procedure                                                     |
|
| This procedure gets calls whenever a window procedure is called
| via WinSendMsg
|
| Notes:
| ------
| 1) we are called by PM, not a PMSPY instance.  This implies that we
|    don't know which PMSPY instance, if any, should SPY this message.
|    That's why we have to scan the complete Spys[] table and route it
|    to all interested SPYees...
|
| Reference: PM Programming Reference: Volume Two,
|            Chapter 10 - "Functions Supplied by Application"
+---------------------------------------------------------------------------*/

VOID EXPENTRY SpySendMsgHookProc(HAB        habSpy_NotUsed,
                                 PSMHSTRUCT pSmh,
                                 BOOL       bTask_NotUsed)
{
  register PdllSPY pSpy;

  /**************************************************************************
  * Check each of the 'active' SPYees to see if interested in this message
  **************************************************************************/

  for(/* Initialize */ pSpy = &Spys[0];      // Start @ first Spyee
      /* While      */ pSpy->fSpyingActive;  // Not at End-Of-List
      /* Iterate    */ pSpy++)               // Move to next Spyee
  {
    /**************************************************************************
    * Determine if this MSG should go to this "active" SPYee:
    *
    * 1) destined for the correct Window or Queue
    * 2) its a MSG within the desired range
    **************************************************************************/

    if ( (pSpy->fTargetIsWindow
          ? pSpy->hwndTarget == pSmh->hwnd
          : pSpy-> hmqTarget == (HMQ)WinQueryWindowULong(pSmh->hwnd,QWL_HMQ)
         ) &&
         ((pSmh->msg >= pSpy->msgLow) && (pSmh->msg <= pSpy->msgHigh)) )
    {
      pSpy->qMsg.hwnd = pSmh->hwnd;             // convert to QMSG data
      pSpy->qMsg.msg  = pSmh->msg;
      pSpy->qMsg.mp1  = pSmh->mp1;
      pSpy->qMsg.mp2  = pSmh->mp2;

      WinSendMsg(pSpy->hwndSpy,
                 MSG_FROM_SPY_HOOK,
                 MPFROMP(&pSpy->qMsg),
                 NULL);
    }
  }
}

/*---------------------------------------------------------------------------
| SpyRegister: register a new PMPSY instance for SPYing
|----------------------------------------------------------------------------
|
| Parms:   hwndSpy.........window to be sent MSGs of SPY activity
|           habSpy.........HAB of SPY window
|           modDLL.........HMODULE of DLL with "hook" procedures
|
| Returns: 0-n.............registered SPY instance (used on other 'Spy' calls)
|          SPY_BAD_SPYEE...unable to register new SPY instance (all instances
|                          already in use)
|
+--------------------------------------------------------------------------*/

SPYEE  EXPENTRY SpyRegister(HWND    hwndSpy,  // window to handle SPY data
                            HAB     habSpy,   // HAB of window
                            HMODULE modPFN)   // DLL handle

{
  register SPYEE idSpy;

  /**************************************************************************
  * Action taken is based on how SPYee are already active
  **************************************************************************/

  switch ( uSpyInstances )
  {
    /**************************************************************************
    * No SPYee is active yet, so set "hooks" into the system
    **************************************************************************/
    case 0:

         WinSetHook(habSpy,
                    NULL,                       // add "hook" to System chain
                    HK_INPUT,                   // "hook" type
                    (PFN)SpyInputHookProc,      // "hook" procedure
                    modPFN);

         WinSetHook(habSpy,
                    NULL,                       // add "hook" to System chain
                    HK_SENDMSG,                 // "hook" type
                    (PFN)SpySendMsgHookProc,    // "hook" procedure
                    modPFN);

         /*** fall through to add new instance ***/

    /**************************************************************************
    * Some SPYees are already active, just add to table
    **************************************************************************/

    default:

         idSpy = uSpyInstances;     // Save this instance # (0-origin)

         uSpyInstances++;           // Bump # active SPYees (1-origin)

         /**************************************************************************
         * Add to Instance-to-SPY data translation table
         **************************************************************************/

         Id2Spys[idSpy] = idSpy;

         /**************************************************************************
         * Add to SPY data table
         **************************************************************************/

         Spys[idSpy]         = defSPY;     // default all fields

         Spys[idSpy]. habSpy =  habSpy;    // fill in specific fields
         Spys[idSpy].hwndSpy = hwndSpy;
         Spys[idSpy].modPFN  = modPFN;

    break;

    /**************************************************************************
    * All SPYee instances are already active
    **************************************************************************/

    case MAX_SPYEES:

         idSpy = SPY_BAD_SPYEE;
    break;
  }

  /**************************************************************************
  * Return the registration result
  **************************************************************************/
  return( idSpy );
}

/*---------------------------------------------------------------------------
| SpyDeRegister: de-register a existing PMPSY instance from SPYing
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of the PMSPY instance to de-register
|
| Returns: TRUE............instance successfully de-registered
|          FALSE...........instance NOT          de-registered (bad ID)
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpyDeRegister(SPYEE   idSpy)    // which SPY instance

{
  register PdllSPY pSpy;
  register USHORT  i;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,FALSE,pSpy);

  /**************************************************************************
  * Action taken is based on how many SPYee are already active
  **************************************************************************/

  switch ( uSpyInstances )
  {
    /**************************************************************************
    * This is the last SPYee....so remove "hooks" from the system
    **************************************************************************/
    case 1:

         WinReleaseHook(pSpy->habSpy,
                        NULL,                    // de"hook" from System chain
                        HK_INPUT,                //   "hook" type
                        (PFN)SpyInputHookProc,   //   "hook" procedure
                        pSpy->modPFN);

         WinReleaseHook(pSpy->habSpy,
                        NULL,                    // de"hook" from System chain
                        HK_SENDMSG,              //   "hook" type
                        (PFN)SpySendMsgHookProc, //   "hook" procedure
                        pSpy->modPFN);

         /*** fall through to delete instance ***/

    /**************************************************************************
    * Some SPYees will remain active, just delete this ID from tables
    **************************************************************************/

    default:

         uSpyInstances--;           // Decrement # active SPYees (1-origin)

         /**************************************************************************
         * Pack all SPY data table items below this one up one slot
         *
         * Note: we're intentionally moving the MAX_SPYEES element since
         *       if exists and is always an EOT element
         *
         **************************************************************************/

         for(/* Initialize */ i = idSpy;       // Start at this ID
             /* While      */ i <= MAX_SPYEES; // While all below not moved
             /* Iterate    */ i++)             // keep marching...
         {
            Spys[i] = Spys[i + 1];
         }

         /**************************************************************************
         * Remove from Instance-to-SPY data translation table
         **************************************************************************/

         Id2Spys[idSpy] = ID_NOT_DEFINED;

         /**************************************************************************
         * Adjust Id2Spy[] to locate 'Packed' SPY data table
         *
         * - we only do this if Spys[] data element was below the
         *   SPYee we've just deleted and packed up a slot
         **************************************************************************/

         for(/* Initialize */ i = 0;           // Start at first ID
             /* While      */ i < MAX_SPYEES;  // While all not processed
             /* Iterate    */ i++)             // keep marching...
         {
            if ( (Id2Spys[i] != ID_NOT_DEFINED)  &&   // defined?
                 (Id2Spys[i] >  idSpy) )              // below deleted ID?
              Id2Spys[i]--;                           // yes: adjust 1 slot
         }

    break;
  }

  /**************************************************************************
  * Return that the ID was successfully de-registered
  **************************************************************************/
  return( TRUE );
}

/*---------------------------------------------------------------------------
| SpySetTarget: set new target window and queue
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|          hwndSpy.........target Window to now SPY
|           hmqSpy.........target Queue  to now SPY
|
| Returns: TRUE............target successfully set
|          FALSE...........target NOT          set (bad ID)
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpySetTarget(register SPYEE idSpy,      // SPY instance
                                    HWND  hwndTarget, // Window to SPY
                                    HMQ   hmqTarget)  // Queue  to SPY

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,FALSE,pSpy);

  /**************************************************************************
  * Set instance's new Window and Queue
  **************************************************************************/

  pSpy->hwndTarget = hwndTarget;
  pSpy-> hmqTarget =  hmqTarget;

  /**************************************************************************
  * Return that new targets successfully set
  **************************************************************************/
  return( TRUE );
}

/*---------------------------------------------------------------------------
| SpyUnSetTarget: unset instance's target window and queue
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of the PMSPY instance to Un-Set
|
| Returns: TRUE............target successfully Un-Set
|          FALSE...........target NOT          Un-Set (bad ID)
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpyUnSetTarget(register SPYEE idSpy) // SPY instance

{
  /**************************************************************************
  * Set instance's new Window and Queue to "invalid" values
  **************************************************************************/

  return( SpySetTarget(idSpy, SPY_BAD_HWND, SPY_BAD_HMQ) );
}

/*---------------------------------------------------------------------------
| SpyQueryTargetWindow: query instance's target window
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|
| Returns: !SPY_BAD_HWND...window successfully queried
|          SPY_BAD_HWND....window NOT          queried (bad ID)
|
+--------------------------------------------------------------------------*/

HWND EXPENTRY SpyQueryTargetWindow(register SPYEE idSpy) // SPY instance

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,SPY_BAD_HWND,pSpy);

  /**************************************************************************
  * Return instance's Window
  **************************************************************************/

  return( pSpy->hwndTarget );
}

/*---------------------------------------------------------------------------
| SpyQueryTargetQueue: query instance's target queue
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|
| Returns: !SPY_BAD_HMQ....queue successfully queried
|          SPY_BAD_HMQ.....queue NOT          queried (bad ID)
|
+--------------------------------------------------------------------------*/

HMQ EXPENTRY SpyQueryTargetQueue(register SPYEE idSpy) // SPY instance

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,SPY_BAD_HMQ,pSpy);

  /**************************************************************************
  * Return instance's Queue
  **************************************************************************/

  return( pSpy->hmqTarget );
}


/*---------------------------------------------------------------------------
| SpySetTargetIsWindow: set that this instance is SPYing the Window/Queue
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|          fTargetIsWindow.TRUE=SPY on window, FALSE=SPY on queue
|
| Returns: TRUE............successful
|          FALSE...........NOT successful (bad ID)
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpySetTargetIsWindow(register SPYEE idSpy,      // SPY instance
                                            BOOL  fTargetIsWindow)  // SPYing on window?

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,FALSE,pSpy);

  /**************************************************************************
  * Return instance's Queue
  **************************************************************************/

  pSpy->fTargetIsWindow = fTargetIsWindow;

  /**************************************************************************
  * Return that is worked!
  **************************************************************************/

  return( TRUE );
}

/*---------------------------------------------------------------------------
| SpyQueryTargetIsWindow: query if this instance is SPYing the Window/Queue
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|
| Returns: TRUE............SPYing window
|          FALSE...........SPYing queue -or- bad ID
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpyQueryTargetIsWindow(register SPYEE idSpy) // SPY instance

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,FALSE,pSpy);

  /**************************************************************************
  * Return instance's SPY state
  **************************************************************************/

  return( pSpy->fTargetIsWindow );
}

/*---------------------------------------------------------------------------
| SpySetTargetMsgRange: set instance's MSG interest range
|----------------------------------------------------------------------------
|
| Parms:   idSpy...........ID of desired PMSPY instance
|          msgLow..........low  MSG in inclusive range of desired MSGs
|          msgHigh.........high MSG in inclusive range of desired MSGs
|
| Returns: TRUE............MSG interest range set
|          FALSE...........MSG interest range NOT set (bad ID or MSG range)
|
+--------------------------------------------------------------------------*/

BOOL EXPENTRY SpySetTargetMsgRange(register SPYEE idSpy, // SPY instance
                                   MSG   msgLow,     // lowest  MSG in range
                                   MSG   msgHigh)    // highest MSG in range

{
  register PdllSPY pSpy;

  /**************************************************************************
  * Validate that the ID is valid - if not immediate return is performed
  *                               - if so, pSpy will locate it's data
  **************************************************************************/

  IF_BAD_ID_RETURN_ELSE_SET(idSpy,FALSE,pSpy);

  /**************************************************************************
  * Validate specified MSG range: low must really be lower or the same!
  **************************************************************************/

  if ( msgLow <= msgHigh )
  {
    pSpy->msgLow  = msgLow;
    pSpy->msgHigh = msgHigh;

    return( TRUE );
  }
  else
    return( FALSE );
}

/*---------------------------------------------------------------------------
| SpyDllVersion: Return the Version of the SpyDll Module
|----------------------------------------------------------------------------
|
| Parms:   NONE
|
| Returns: numeric version ID of this PMSPY support ID
|
+--------------------------------------------------------------------------*/

USHORT EXPENTRY SpyDllVersion(VOID)
{
  return( DLLVERSION );
}

/*---------------------------------------------------------------------------
| SpyQueryDllRegisterMax: Return maximum number of concurrent PMSPY instances
|----------------------------------------------------------------------------
|
| Parms:   NONE
|
| Returns: maximum number of PMSPY instances that can be registered concurrently
|
+--------------------------------------------------------------------------*/

USHORT EXPENTRY SpyQueryDllRegisterMax(VOID)
{
  return( MAX_SPYEES );
}
